import React, { useState } from 'react';
import { NavLink } from 'react-router-dom';

interface HeaderProps {
    isLoggedIn: boolean;
    username: string | null;
    onLogout: () => void;
}

const Header: React.FC<HeaderProps> = ({ isLoggedIn, username, onLogout }) => {
    const [isMenuOpen, setIsMenuOpen] = useState(false);

    const linkStyle = "text-gray-300 hover:bg-gray-700 hover:text-white px-3 py-2 rounded-md text-sm font-medium transition-colors block md:inline-block";
    const activeLinkStyle = {
        backgroundColor: '#4f46e5',
        color: 'white'
    };

    const navLinks = (
        <>
            <NavLink to="/" style={({ isActive }) => isActive ? activeLinkStyle : undefined} className={linkStyle} onClick={() => setIsMenuOpen(false)}>خانه</NavLink>
            <NavLink to="/games" style={({ isActive }) => isActive ? activeLinkStyle : undefined} className={linkStyle} onClick={() => setIsMenuOpen(false)}>لیست بازی‌ها</NavLink>
            <NavLink to="/leaderboard" style={({ isActive }) => isActive ? activeLinkStyle : undefined} className={linkStyle} onClick={() => setIsMenuOpen(false)}>جدول امتیازات</NavLink>
            <NavLink to="/rules" style={({ isActive }) => isActive ? activeLinkStyle : undefined} className={linkStyle} onClick={() => setIsMenuOpen(false)}>قوانین</NavLink>
        </>
    );
    
    return (
        <header className="bg-gray-800 shadow-lg sticky top-0 z-50">
            <div className="container mx-auto px-4">
                <div className="flex items-center justify-between h-16">
                    <div className="flex items-center">
                        <NavLink to="/" className="text-white text-xl font-bold">
                           🎮 Game Zone
                        </NavLink>
                        <div className="hidden md:flex items-baseline space-x-4 mr-4">
                            {navLinks}
                        </div>
                    </div>
                    <div className="hidden md:flex items-center">
                        {isLoggedIn ? (
                            <>
                                <span className="text-gray-300 text-sm ml-4">خوش آمدی, {username}!</span>
                                <button
                                    onClick={onLogout}
                                    className="bg-red-600 hover:bg-red-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors"
                                >
                                    خروج
                                </button>
                            </>
                        ) : (
                            <div className="space-x-2 space-x-reverse">
                                <NavLink to="/login" className="bg-indigo-600 hover:bg-indigo-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors">ورود</NavLink>
                                <NavLink to="/register" className="text-gray-300 hover:bg-gray-700 hover:text-white px-3 py-2 rounded-md text-sm font-medium transition-colors">ثبت نام</NavLink>
                            </div>
                        )}
                    </div>
                    <div className="md:hidden flex items-center">
                        <button onClick={() => setIsMenuOpen(!isMenuOpen)} className="text-gray-300 hover:text-white focus:outline-none focus:text-white">
                            <svg className="h-6 w-6 fill-current" viewBox="0 0 24 24">
                                {isMenuOpen ? (
                                    <path fillRule="evenodd" clipRule="evenodd" d="M18.278 16.864a1 1 0 0 1-1.414 1.414l-4.829-4.828-4.828 4.828a1 1 0 0 1-1.414-1.414l4.828-4.829-4.828-4.828a1 1 0 0 1 1.414-1.414l4.829 4.828 4.828-4.828a1 1 0 1 1 1.414 1.414l-4.828 4.829 4.828 4.828z" />
                                ) : (
                                    <path fillRule="evenodd" d="M4 5h16a1 1 0 0 1 0 2H4a1 1 0 1 1 0-2zm0 6h16a1 1 0 0 1 0 2H4a1 1 0 0 1 0-2zm0 6h16a1 1 0 0 1 0 2H4a1 1 0 0 1 0-2z" />
                                )}
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
            {isMenuOpen && (
                <div className="md:hidden px-2 pt-2 pb-3 space-y-1 sm:px-3">
                    {navLinks}
                    <div className="border-t border-gray-700 my-2"></div>
                     {isLoggedIn ? (
                            <div className="px-2 py-2">
                                <span className="text-gray-300 text-sm block mb-2">خوش آمدی, {username}!</span>
                                <button
                                    onClick={() => { onLogout(); setIsMenuOpen(false); }}
                                    className="w-full text-left bg-red-600 hover:bg-red-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors"
                                >
                                    خروج
                                </button>
                            </div>
                        ) : (
                            <div className="flex items-center space-x-2 space-x-reverse px-2 py-2">
                                <NavLink to="/login" className="flex-1 text-center bg-indigo-600 hover:bg-indigo-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors" onClick={() => setIsMenuOpen(false)}>ورود</NavLink>
                                <NavLink to="/register" className="flex-1 text-center text-gray-300 hover:bg-gray-700 hover:text-white px-3 py-2 rounded-md text-sm font-medium transition-colors" onClick={() => setIsMenuOpen(false)}>ثبت نام</NavLink>
                            </div>
                        )}
                </div>
            )}
        </header>
    );
};

export default Header;