import { getScores, saveScores } from '../services/storageService';
import type { Score, User } from '../types';
import { BOT_USER } from '../constants';

// A helper to ensure user exists in scores array and return their score object
const ensureUserInScores = (scores: Score[], user: User): Score => {
    let userScore = scores.find(s => s.userId === user.id);
    if (!userScore) {
        userScore = { userId: user.id, username: user.username, wins: 0, losses: 0, draws: 0, points: 0 };
        scores.push(userScore);
    } else if (user.id === BOT_USER.id && userScore.username !== BOT_USER.username) {
        // Ensure bot username is always correct, in case it was somehow changed
        userScore.username = BOT_USER.username;
    }
    return userScore;
}

export const useScoreUpdater = () => {
  /**
   * Updates scores for two players based on a game result.
   * @param player1 - The first player object (typically the logged-in user).
   * @param player2 - The second player object (the opponent or the bot).
   * @param result - The outcome of the game.
   */
  const updateScores = (player1: User, player2: User, result: 'player1_wins' | 'player2_wins' | 'draw') => {
    // A copy is needed to avoid mutation issues before saving
    let scores = [...getScores()];
    
    const player1Score = ensureUserInScores(scores, player1);
    const player2Score = ensureUserInScores(scores, player2);

    if (result === 'player1_wins') {
      player1Score.wins += 1;
      player1Score.points += 3;
      player2Score.losses += 1;
    } else if (result === 'player2_wins') {
      player2Score.wins += 1;
      player2Score.points += 3;
      player1Score.losses += 1;
    } else if (result === 'draw') {
      player1Score.draws += 1;
      player1Score.points += 1;
      player2Score.draws += 1;
      player2Score.points += 1;
    }

    saveScores(scores);
  };

  return { updateScores };
};
