import React, { useState, useContext, useEffect, useCallback, useMemo } from 'react';
import { useNavigate, Link, useLocation } from 'react-router-dom';
import { AuthContext } from '../App';
import { useScoreUpdater } from '../hooks/useScoreUpdater';
import type { User } from '../types';
import { BOT_USER } from '../constants';
import Chat from '../components/Chat';

type Player = 'white' | 'black';

interface Point {
    player: Player | null;
    count: number;
}

// Constants
const TOTAL_POINTS = 24;
const CHECKERS_PER_PLAYER = 15;
const PLAYER_WHITE: Player = 'white'; // Human player
const PLAYER_BLACK: Player = 'black'; // AI opponent

const getInitialPoints = (): Point[] => {
    const points: Point[] = Array(TOTAL_POINTS).fill(null).map(() => ({ player: null, count: 0 }));
    points[0]  = { player: PLAYER_BLACK, count: 2 };
    points[5]  = { player: PLAYER_WHITE, count: 5 };
    points[7]  = { player: PLAYER_WHITE, count: 3 };
    points[11] = { player: PLAYER_BLACK, count: 5 };
    points[12] = { player: PLAYER_WHITE, count: 5 };
    points[16] = { player: PLAYER_BLACK, count: 3 };
    points[18] = { player: PLAYER_BLACK, count: 5 };
    points[23] = { player: PLAYER_WHITE, count: 2 };
    return points;
};

const Die = ({ value }: { value: number | 0 }) => {
    if (value === 0) return <div className="w-10 h-10 bg-white/20 rounded-md shadow-inner animate-pulse" />;
    return <div className={`w-10 h-10 bg-stone-100 rounded-md shadow-lg flex items-center justify-center text-4xl font-bold text-black dice-face-${value}`}></div>;
};

const Checker = ({ player, isSelected, isHit }: { player: Player, isSelected?: boolean, isHit?: boolean }) => {
    const baseClasses = `w-full aspect-square rounded-full transition-all duration-200 transform`;
    const playerClasses = player === PLAYER_WHITE 
        ? 'bg-gradient-to-br from-stone-100 to-stone-300 shadow-[inset_0_2px_4px_rgba(0,0,0,0.4)] border border-stone-500' 
        : 'bg-gradient-to-br from-gray-500 to-gray-800 shadow-[inset_0_2px_4px_rgba(0,0,0,0.6)] border border-black';
    const selectedClasses = isSelected ? 'ring-4 ring-yellow-400 ring-offset-2 ring-offset-black/50 scale-110 -translate-y-1 z-20' : '';
    const hitClasses = isHit ? 'opacity-80' : 'shadow-md';
    
    return <div className={`${baseClasses} ${playerClasses} ${selectedClasses} ${hitClasses}`} />;
};

const BackgammonGame: React.FC = () => {
    const [points, setPoints] = useState<Point[]>(Array(TOTAL_POINTS).fill(null).map(() => ({ player: null, count: 0 })));
    const [bar, setBar] = useState({ [PLAYER_WHITE]: 0, [PLAYER_BLACK]: 0 });
    const [borneOff, setBorneOff] = useState({ [PLAYER_WHITE]: 0, [PLAYER_BLACK]: 0 });
    const [turn, setTurn] = useState<Player>(PLAYER_WHITE);
    const [dice, setDice] = useState<[number, number]>([0, 0]);
    const [movesLeft, setMovesLeft] = useState<number[]>([]);
    const [gameState, setGameState] = useState<'SETUP' | 'ROLL' | 'MOVE' | 'GAMEOVER'>('SETUP');
    const [selectedPoint, setSelectedPoint] = useState<number | 'bar' | null>(null);
    const [possibleMoves, setPossibleMoves] = useState<Map<number, number>>(new Map());
    const [gameOverMessage, setGameOverMessage] = useState<string | null>(null);
    const [isInitialSetup, setIsInitialSetup] = useState(true);

    const auth = useContext(AuthContext);
    const navigate = useNavigate();
    const location = useLocation();
    const { updateScores } = useScoreUpdater();

    const opponent = useMemo(() => (location.state?.opponent as User | undefined) || auth?.otherUser, [location.state, auth?.otherUser]);

    useEffect(() => {
        if (gameState === 'SETUP' && isInitialSetup) {
            setPoints(Array(TOTAL_POINTS).fill(null).map(() => ({ player: null, count: 0 })));

            const initialPointsConfig = getInitialPoints();
            const pointsWithCheckers = initialPointsConfig
                .map((p, i) => ({ ...p, index: i }))
                .filter(p => p.count > 0)
                .sort((a, b) => a.index - b.index);

            let delay = 100;
            pointsWithCheckers.forEach(pointConfig => {
                setTimeout(() => {
                    setPoints(prevPoints => {
                        const newPoints = prevPoints.map(p => ({ ...p }));
                        newPoints[pointConfig.index] = {
                            player: pointConfig.player,
                            count: pointConfig.count
                        };
                        return newPoints;
                    });
                }, delay);
                delay += 75;
            });

            setTimeout(() => {
                setIsInitialSetup(false);
                setGameState('ROLL');
            }, delay + 400);
        }
    }, [gameState, isInitialSetup]);


    const changeTurn = useCallback(() => {
        setTurn(currentTurn => currentTurn === PLAYER_WHITE ? PLAYER_BLACK : PLAYER_WHITE);
        setGameState('ROLL');
        setDice([0, 0]);
        setMovesLeft([]);
        setSelectedPoint(null);
        setPossibleMoves(new Map());
    }, []);

    const canBearOff = useCallback((player: Player, currentPoints: Point[], currentBar: {white: number, black: number}): boolean => {
        if (currentBar[player] > 0) return false;
        
        const homeRange = player === PLAYER_WHITE ? {start: 0, end: 5} : {start: 18, end: 23};
        let checkersOutsideHome = 0;
        
        for (let i = 0; i < TOTAL_POINTS; i++) {
            if (currentPoints[i].player === player) {
                const isOutside = player === PLAYER_WHITE ? i > homeRange.end : i < homeRange.start;
                if (isOutside) {
                    checkersOutsideHome += currentPoints[i].count;
                }
            }
        }
        return checkersOutsideHome === 0;
    }, []);

    const getValidMovesForDie = useCallback((from: number | 'bar', die: number, player: Player, currentPoints: Point[], currentBar: {white: number, black: number}): number | null => {
        const opponentPlayer = player === PLAYER_WHITE ? PLAYER_BLACK : PLAYER_WHITE;

        if (from === 'bar') {
            const targetIndex = player === PLAYER_WHITE ? 24 - die : die - 1;
            const targetPoint = currentPoints[targetIndex];
            if (targetPoint.player === opponentPlayer && targetPoint.count > 1) return null;
            return targetIndex;
        }

        const isBearingOffPossible = canBearOff(player, currentPoints, currentBar);
        if (isBearingOffPossible) {
            const exactBearOffIndex = player === PLAYER_WHITE ? die - 1 : 24 - die;
            if (from === exactBearOffIndex) {
                return player === PLAYER_WHITE ? -1 : TOTAL_POINTS; // -1 for white bear off, 24 for black
            }
            
            const homeIndices = player === PLAYER_WHITE ? [5,4,3,2,1,0] : [18,19,20,21,22,23];
            const playerCheckersInHome = homeIndices.filter(i => currentPoints[i].player === player);

            if (playerCheckersInHome.length > 0) {
                 const furthestCheckerIndex = player === PLAYER_WHITE ? Math.max(...playerCheckersInHome) : Math.min(...playerCheckersInHome);
                 if (from === furthestCheckerIndex) {
                    const isRollHigher = player === PLAYER_WHITE ? (die - 1 > furthestCheckerIndex) : (24 - die < furthestCheckerIndex);
                    if (isRollHigher) return player === PLAYER_WHITE ? -1 : TOTAL_POINTS;
                 }
            }
        }
        
        const targetIndex = player === PLAYER_WHITE ? from - die : from + die;
        if (targetIndex < 0 || targetIndex >= TOTAL_POINTS) return null;

        const targetPoint = currentPoints[targetIndex];
        if (targetPoint.player === opponentPlayer && targetPoint.count > 1) return null;

        return targetIndex;
    }, [canBearOff]);
    
    const hasAnyMoves = useCallback((player: Player, diceValues: number[], currentPoints: Point[], currentBar: {white: number, black: number}): boolean => {
        const sources = currentBar[player] > 0 ? ['bar'] as const : currentPoints.map((p, i) => i).filter(i => currentPoints[i].player === player);
        for (const from of sources) {
            for (const die of diceValues) {
                if (getValidMovesForDie(from, die, player, currentPoints, currentBar) !== null) return true;
            }
        }
        return false;
    }, [getValidMovesForDie]);

    const executeMove = useCallback((from: number | 'bar', to: number, die: number, player: Player) => {
        const newPoints = points.map(p => ({...p}));
        const newBar = { ...bar };
        const newBorneOff = { ...borneOff };
        
        if (from === 'bar') {
            newBar[player]--;
        } else {
            newPoints[from].count--;
            if (newPoints[from].count === 0) newPoints[from].player = null;
        }

        if (to === -1 || to === TOTAL_POINTS) { // Bear-off
            newBorneOff[player]++;
        } else {
            const opponentPlayer = player === PLAYER_WHITE ? PLAYER_BLACK : PLAYER_WHITE;
            if (newPoints[to].player === opponentPlayer) {
                newBar[opponentPlayer]++;
                newPoints[to].player = player;
                newPoints[to].count = 1;
            } else {
                newPoints[to].player = player;
                newPoints[to].count++;
            }
        }
        
        const remainingMoves = [...movesLeft];
        const dieIndex = remainingMoves.indexOf(die);
        if (dieIndex > -1) remainingMoves.splice(dieIndex, 1);
        
        setPoints(newPoints);
        setBar(newBar);
        setBorneOff(newBorneOff);
        setSelectedPoint(null);
        setPossibleMoves(new Map());
        
        if (newBorneOff[player] === CHECKERS_PER_PLAYER) {
            setGameState('GAMEOVER');
            const winnerUsername = player === PLAYER_WHITE ? auth?.user?.username : opponent?.username;
            setGameOverMessage(`${winnerUsername} برنده شد`);
            if (auth.user && opponent) {
                updateScores(auth.user, opponent, player === PLAYER_WHITE ? 'player1_wins' : 'player2_wins');
            }
        } else if (remainingMoves.length === 0 || !hasAnyMoves(player, remainingMoves, newPoints, newBar)) {
            setMovesLeft([]);
            setTimeout(changeTurn, 500);
        } else {
            setMovesLeft(remainingMoves);
        }
    }, [points, bar, borneOff, movesLeft, auth, opponent, updateScores, hasAnyMoves, changeTurn]);
    
    const handlePointClick = (index: number | 'bar' | 'bear-off-white' | 'bear-off-black') => {
        if (turn !== PLAYER_WHITE || gameState !== 'MOVE' || gameOverMessage) return;
    
        const bearOffTarget = -1;
        if (index === 'bear-off-white' && possibleMoves.has(bearOffTarget)) {
            const die = possibleMoves.get(bearOffTarget)!;
            executeMove(selectedPoint!, bearOffTarget, die, PLAYER_WHITE);
            return;
        }

        if (selectedPoint !== null && typeof index === 'number' && possibleMoves.has(index)) {
            const die = possibleMoves.get(index)!;
            executeMove(selectedPoint, index, die, PLAYER_WHITE);
            return;
        }

        if (selectedPoint === index) {
            setSelectedPoint(null);
            setPossibleMoves(new Map());
            return;
        }

        if (typeof index !== 'number' && index !== 'bar') return;
        
        if (bar[PLAYER_WHITE] > 0) {
            if (index !== 'bar') return;
        } else {
            if (index === 'bar' || points[index as number].player !== PLAYER_WHITE) return;
        }

        const newPossibleMoves = new Map<number, number>();
        const uniqueDice = [...new Set(movesLeft)];
        for (const die of uniqueDice) {
            const target = getValidMovesForDie(index, die, PLAYER_WHITE, points, bar);
            if (target !== null) {
                newPossibleMoves.set(target, die);
            }
        }

        if (newPossibleMoves.size > 0) {
            setSelectedPoint(index);
            setPossibleMoves(newPossibleMoves);
        } else {
            setSelectedPoint(null);
            setPossibleMoves(new Map());
        }
    };

    const rollDice = () => {
        if (gameState !== 'ROLL' || turn !== PLAYER_WHITE || gameOverMessage) return;
        const d1 = Math.floor(Math.random() * 6) + 1;
        const d2 = Math.floor(Math.random() * 6) + 1;
        setDice([d1, d2]);
        const newMoves = d1 === d2 ? [d1, d1, d1, d1] : [d1, d2];
        if (hasAnyMoves(PLAYER_WHITE, newMoves, points, bar)) {
            setMovesLeft(newMoves);
            setGameState('MOVE');
        } else {
            setMovesLeft([]);
            setTimeout(changeTurn, 1500);
        }
    };

    useEffect(() => {
        if (turn === PLAYER_BLACK && (gameState === 'ROLL' || gameState === 'MOVE') && !gameOverMessage) {
             if (gameState === 'ROLL') {
                setTimeout(() => {
                    const d1 = Math.floor(Math.random() * 6) + 1;
                    const d2 = Math.floor(Math.random() * 6) + 1;
                    setDice([d1, d2]);
                    const newMoves = d1 === d2 ? [d1, d1, d1, d1] : [d1, d2];
                    if (hasAnyMoves(PLAYER_BLACK, newMoves, points, bar)) {
                        setMovesLeft(newMoves);
                        setGameState('MOVE');
                    } else {
                        setMovesLeft([]);
                        setTimeout(changeTurn, 1500);
                    }
                }, 1000);
            } else if (gameState === 'MOVE' && movesLeft.length > 0) {
                setTimeout(() => {
                    const allPossibleMoves: { from: number | 'bar', to: number, die: number }[] = [];
                    const sources = bar[PLAYER_BLACK] > 0 ? ['bar'] as const : points.map((p, i) => i).filter(i => points[i].player === PLAYER_BLACK);

                    for (const from of sources) for (const die of [...new Set(movesLeft)]) {
                        const to = getValidMovesForDie(from, die, PLAYER_BLACK, points, bar);
                        if (to !== null) allPossibleMoves.push({ from, to, die });
                    }
                    
                    // Simple AI: prioritize bearing off, then hitting, then moving forward
                    const bestMove = allPossibleMoves.sort((a, b) => {
                        const aIsBearOff = a.to === TOTAL_POINTS;
                        const bIsBearOff = b.to === TOTAL_POINTS;
                        if (aIsBearOff !== bIsBearOff) return aIsBearOff ? -1 : 1;

                        const aIsHit = a.to < TOTAL_POINTS && points[a.to].player === PLAYER_WHITE;
                        const bIsHit = b.to < TOTAL_POINTS && points[b.to].player === PLAYER_WHITE;
                        if (aIsHit !== bIsHit) return aIsHit ? -1 : 1;

                        return a.to - b.to; // Move pieces further away from start
                    })[0];
                    
                    if (bestMove) {
                        executeMove(bestMove.from, bestMove.to, bestMove.die, PLAYER_BLACK);
                    }
                }, 800);
            }
        }
    }, [turn, gameState, gameOverMessage, points, bar, movesLeft, hasAnyMoves, getValidMovesForDie, changeTurn, executeMove]);

    const resetGame = useCallback(() => {
        setBar({ [PLAYER_WHITE]: 0, [PLAYER_BLACK]: 0 });
        setBorneOff({ [PLAYER_WHITE]: 0, [PLAYER_BLACK]: 0 });
        setTurn(PLAYER_WHITE);
        setDice([0, 0]);
        setMovesLeft([]);
        setSelectedPoint(null);
        setPossibleMoves(new Map());
        setGameOverMessage(null);
        setGameState('SETUP');
        setIsInitialSetup(true);
    }, []);

    const renderPoint = (pointIndex: number) => {
        const point = points[pointIndex];
        const isTop = pointIndex >= 12;
        const isPossible = possibleMoves.has(pointIndex);
        const isSelected = selectedPoint === pointIndex;
        
        return (
            <div className="relative h-full w-full cursor-pointer group" onClick={() => handlePointClick(pointIndex)}>
                <svg viewBox="0 0 100 400" className={`w-full h-full ${!isTop ? 'rotate-180' : ''}`} preserveAspectRatio="none">
                    <polygon points="0,0 100,0 50,400" className={(pointIndex + (isTop ? 1 : 0)) % 2 === 0 ? 'fill-stone-300' : 'fill-stone-500'} />
                </svg>
                <div className={`absolute inset-0 flex p-1 ${isTop ? 'flex-col justify-start' : 'flex-col-reverse justify-start'}`}>
                    {[...Array(point.count)].map((_, i) => (
                        <div key={i} className={`w-[90%] mx-auto ${isTop ? 'checker-wrapper-top' : 'checker-wrapper-bottom'} ${i > 0 ? (isTop ? '-mt-[75%]' : '-mb-[75%]') : ''}`} style={{zIndex: isTop ? 5-i : i}}>
                            <Checker player={point.player!} isSelected={isSelected && i === point.count - 1} />
                        </div>
                    ))}
                </div>
                 {isPossible && <div className={`absolute ${isTop ? 'top-2' : 'bottom-2'} left-1/2 -translate-x-1/2 w-4 h-4 bg-green-400 rounded-full animate-pulse ring-2 ring-white/50`} />}
            </div>
        );
    };
    
    if (!auth?.user || !opponent) {
        return <div className="flex-grow flex items-center justify-center text-center p-8">در حال آماده‌سازی بازی...</div>;
    }

    return (
        <div className="flex-grow flex flex-col items-center justify-start p-2 sm:p-4 bg-gray-900 text-white font-sans">
            <style>{`
              @keyframes checker-enter-top {
                from { opacity: 0; transform: scale(0.3) translateY(-40px) rotate(-45deg); }
                to { opacity: 1; transform: scale(1) translateY(0) rotate(0deg); }
              }
              @keyframes checker-enter-bottom {
                from { opacity: 0; transform: scale(0.3) translateY(40px) rotate(45deg); }
                to { opacity: 1; transform: scale(1) translateY(0) rotate(0deg); }
              }
              .checker-wrapper-top {
                animation: checker-enter-top 0.4s ease-out forwards;
              }
              .checker-wrapper-bottom {
                animation: checker-enter-bottom 0.4s ease-out forwards;
              }
              .dice-face-1 { background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 10 10'%3E%3Ccircle cx='5' cy='5' r='1.5' fill='black'/%3E%3C/svg%3E"); }
              .dice-face-2 { background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 10 10'%3E%3Ccircle cx='2.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='7.5' r='1.5' fill='black'/%3E%3C/svg%3E"); }
              .dice-face-3 { background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 10 10'%3E%3Ccircle cx='2.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='5' cy='5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='7.5' r='1.5' fill='black'/%3E%3C/svg%3E"); }
              .dice-face-4 { background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 10 10'%3E%3Ccircle cx='2.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='2.5' cy='7.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='7.5' r='1.5' fill='black'/%3E%3C/svg%3E"); }
              .dice-face-5 { background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 10 10'%3E%3Ccircle cx='2.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='5' cy='5' r='1.5' fill='black'/%3E%3Ccircle cx='2.5' cy='7.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='7.5' r='1.5' fill='black'/%3E%3C/svg%3E"); }
              .dice-face-6 { background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 10 10'%3E%3Ccircle cx='2.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='2.5' r='1.5' fill='black'/%3E%3Ccircle cx='2.5' cy='5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='5' r='1.5' fill='black'/%3E%3Ccircle cx='2.5' cy='7.5' r='1.5' fill='black'/%3E%3Ccircle cx='7.5' cy='7.5' r='1.5' fill='black'/%3E%3C/svg%3E"); }
              .dice-face-1, .dice-face-2, .dice-face-3, .dice-face-4, .dice-face-5, .dice-face-6 { background-size: 80%; background-position: center; background-repeat: no-repeat; }
            `}</style>
            <h1 className="text-4xl font-bold mb-4">🎲 تخته نرد</h1>
            <div className="flex flex-col lg:flex-row items-start gap-8 w-full max-w-screen-lg mx-auto">
                 <div className="w-full lg:w-72 flex-shrink-0 order-2 lg:order-1 space-y-4">
                     <div className={`bg-gray-800 p-4 rounded-lg shadow-xl border-2 transition-colors ${turn === PLAYER_WHITE && gameState !== 'SETUP' ? 'border-yellow-400' : 'border-gray-700'}`}>
                        <h3 className="text-xl font-bold mb-2 text-stone-300">{auth.user.username} (سفید)</h3>
                    </div>
                     <div className={`bg-gray-800 p-4 rounded-lg shadow-xl border-2 transition-colors ${turn === PLAYER_BLACK && gameState !== 'SETUP' ? 'border-yellow-400' : 'border-gray-700'}`}>
                        <h3 className="text-xl font-bold mb-2 text-gray-800 bg-white/80 rounded px-1 inline-block">{opponent.username} (سیاه)</h3>
                    </div>
                     <Chat currentUser={auth.user} opponent={opponent} />
                </div>

                <div className="flex-grow flex flex-col items-center order-1 lg:order-2">
                    <div className="bg-gradient-to-b from-amber-900 to-amber-700 p-2 rounded-lg shadow-2xl">
                        <div className="bg-teal-900 p-2 flex gap-2 w-full max-w-3xl aspect-[1.1/1]">
                            {/* Board Left */}
                            <div className="flex-1 grid grid-cols-6 h-full">
                                {[...Array(6)].map((_, i) => renderPoint(12 + i))}
                                {[...Array(6)].map((_, i) => renderPoint(11 - i))}
                            </div>
                            {/* Bar */}
                            <div className="w-12 bg-amber-800 h-full flex flex-col justify-between py-2 px-1 cursor-pointer" onClick={() => handlePointClick('bar')}>
                                <div className="flex-1 flex flex-col-reverse justify-start">
                                    {[...Array(bar[PLAYER_WHITE])].map((_, i) => <div key={`bw${i}`} className="w-full -mb-[70%]"><Checker player={PLAYER_WHITE} isHit isSelected={selectedPoint === 'bar' && i === bar[PLAYER_WHITE]-1} /></div>)}
                                </div>
                                <div className="flex-1 flex flex-col justify-start">
                                    {[...Array(bar[PLAYER_BLACK])].map((_, i) => <div key={`bb${i}`} className="w-full -mt-[70%]"><Checker player={PLAYER_BLACK} isHit /></div>)}
                                </div>
                            </div>
                            {/* Board Right */}
                             <div className="flex-1 grid grid-cols-6 h-full">
                                {[...Array(6)].map((_, i) => renderPoint(18 + i))}
                                {[...Array(6)].map((_, i) => renderPoint(5 - i))}
                            </div>
                        </div>
                    </div>
                    {/* Controls & Bear-off Area */}
                    <div className="w-full max-w-3xl flex justify-between items-center mt-4 p-4 bg-gray-800 rounded-lg">
                        <div className={`relative w-24 h-full bg-gray-700/50 rounded p-2 flex flex-col-reverse gap-1 cursor-pointer border-2 ${possibleMoves.has(-1) ? 'border-green-400' : 'border-transparent'}`} onClick={() => handlePointClick('bear-off-white')}>
                             <span className="absolute top-1 right-1 text-xs bg-black/30 px-1 rounded">{borneOff[PLAYER_WHITE]}/{CHECKERS_PER_PLAYER}</span>
                             {[...Array(borneOff[PLAYER_WHITE])].map((_,i)=><div key={`bo-w-${i}`} className="w-full h-2 bg-stone-300 rounded-sm"></div>)}
                        </div>
                         <div className="flex flex-col items-center">
                            {gameOverMessage ? (
                                <div className='text-center'>
                                    <p className="text-2xl font-bold text-yellow-400">{gameOverMessage}</p>
                                    <button onClick={resetGame} className="mt-2 bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-2 px-6 rounded-full text-lg transition-transform transform hover:scale-105 shadow-lg">
                                        بازی مجدد
                                    </button>
                                </div>
                            ) : (
                                <>
                                    <h3 className="font-bold text-lg mb-2 min-h-[28px]">
                                        {gameState === 'SETUP' ? 'در حال چیدن مهره‌ها...' : (turn === PLAYER_WHITE ? "نوبت شما" : (opponent?.username + ' در حال فکر کردن...'))}
                                    </h3>
                                    <div className="flex gap-4 items-center h-12">
                                        <Die value={dice[0]} />
                                        <Die value={dice[1]} />
                                    </div>
                                    {gameState === 'ROLL' && turn === PLAYER_WHITE && (
                                        <button onClick={rollDice} className="mt-3 bg-green-600 hover:bg-green-700 text-white font-bold py-2 px-6 rounded-full transition-colors">
                                            تاس بیانداز
                                        </button>
                                    )}
                                </>
                            )}
                        </div>
                        <div className={`relative w-24 h-full bg-gray-700/50 rounded p-2 flex flex-col-reverse gap-1 cursor-pointer border-2 ${possibleMoves.has(TOTAL_POINTS) ? 'border-green-400' : 'border-transparent'}`} onClick={() => handlePointClick('bear-off-black')}>
                            <span className="absolute top-1 right-1 text-xs bg-black/30 px-1 rounded">{borneOff[PLAYER_BLACK]}/{CHECKERS_PER_PLAYER}</span>
                             {[...Array(borneOff[PLAYER_BLACK])].map((_,i)=><div key={`bo-b-${i}`} className="w-full h-2 bg-gray-600 rounded-sm"></div>)}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default BackgammonGame;