import React, { useState, useContext, useEffect, useMemo, useCallback } from 'react';
import { useNavigate, Link, useLocation } from 'react-router-dom';
import { AuthContext } from '../App';
import { useScoreUpdater } from '../hooks/useScoreUpdater';
import type { User } from '../types';
import { Chess } from 'chess.js';
import { Chessboard } from 'react-chessboard';
import { BOT_USER } from '../constants';
import Chat from '../components/Chat';

const ChessGame: React.FC = () => {
    const game = useMemo(() => new Chess(), []);
    const [fen, setFen] = useState(game.fen());
    const [gameOver, setGameOver] = useState<{ message: string; reason: string } | null>(null);

    const auth = useContext(AuthContext);
    const navigate = useNavigate();
    const location = useLocation();
    const { updateScores } = useScoreUpdater();

    const opponent = (location.state?.opponent as User | undefined) || auth?.otherUser;
    const isBotGame = opponent?.id === BOT_USER.id;
    // Player is always white ('w')
    const playerColor = 'w';

    useEffect(() => {
        if (opponent && (!auth?.otherUser || auth.otherUser.id !== opponent.id)) {
            auth?.setOtherUser(opponent);
        }
    }, [opponent, auth]);

    useEffect(() => {
        if (auth && !auth.isLoggedIn) {
            navigate('/login');
        }
    }, [auth, navigate]);
    
    const handleGameEnd = useCallback((result: 'player1_wins' | 'player2_wins' | 'draw', reason: string) => {
        if (!auth?.user || !opponent || gameOver) return;

        if (result === 'player1_wins') {
            updateScores(auth.user, opponent, 'player1_wins');
        } else if (result === 'player2_wins') {
            updateScores(auth.user, opponent, 'player2_wins');
        } else {
            updateScores(auth.user, opponent, 'draw');
        }
        setGameOver({ message: `بازی تمام شد.`, reason });
    }, [auth?.user, opponent, gameOver, updateScores]);


    const checkGameEnd = useCallback(() => {
        if (game.isGameOver()) {
            let reason = '';
            let winner = game.turn() === 'w' ? opponent : auth?.user; // If it's white's turn to move but game is over, black won.

            if (game.isCheckmate()) {
                reason = `مات! ${winner?.username} برنده شد.`;
                if (winner?.id === auth?.user?.id) {
                    handleGameEnd('player1_wins', reason);
                } else {
                    handleGameEnd('player2_wins', reason);
                }
            } else if (game.isDraw()) {
                reason = 'بازی مساوی شد.';
                if(game.isStalemate()){
                    reason = 'پات! بازی مساوی شد.'
                } else if (game.isThreefoldRepetition()){
                    reason = 'تکرار سه گانه! بازی مساوی شد.'
                } else if(game.isInsufficientMaterial()){
                    reason = 'مهره کافی برای مات کردن وجود ندارد! بازی مساوی شد.'
                }
                handleGameEnd('draw', reason);
            }
        }
    }, [game, auth?.user, opponent, handleGameEnd]);
    
    // AI move logic
    useEffect(() => {
        if (isBotGame && game.turn() !== playerColor && !gameOver) {
            const timer = setTimeout(() => {
                const moves = game.moves();
                if (moves.length > 0) {
                    const randomMove = moves[Math.floor(Math.random() * moves.length)];
                    game.move(randomMove);
                    setFen(game.fen());
                    checkGameEnd();
                }
            }, 800); // Bot "thinking" delay
            return () => clearTimeout(timer);
        }
    }, [fen, game, isBotGame, playerColor, gameOver, checkGameEnd]);

    function onDrop(sourceSquare: string, targetSquare: string) {
        // Prevent player from moving when it's not their turn, or if game is over
        if (game.turn() !== playerColor || gameOver) {
            return false;
        }
        // Don't allow moves if playing against another human (for now)
        if (!isBotGame && opponent?.id !== auth?.user?.id) {
            // In a real multiplayer scenario, this would be handled by websockets
            // For this version, we only allow playing against the bot
        }

        let move = null;
        try {
            move = game.move({
                from: sourceSquare,
                to: targetSquare,
                promotion: 'q', // always promote to a queen for simplicity
            });
        } catch (e) {
            // illegal move
            return false;
        }

        // if the move was illegal, chess.js returns null
        if (move === null) {
            return false;
        }

        setFen(game.fen());
        checkGameEnd();
        
        return true;
    }

    const playAgain = () => {
       game.reset();
       setFen(game.fen());
       setGameOver(null);
    };

    if (!auth?.user || !opponent) {
        return <div className="flex-grow flex items-center justify-center text-center p-8">در حال آماده‌سازی بازی...</div>;
    }

    const boardWidth = Math.min(window.innerWidth * 0.9, 560);
    
    let turnStatus;
    if (!gameOver) {
        turnStatus = game.turn() === playerColor ? "نوبت شما" : `نوبت ${opponent.username}`;
        if(isBotGame && game.turn() !== playerColor) {
            turnStatus = "کامپیوتر در حال فکر کردن است...";
        }
    } else {
        turnStatus = "";
    }


    return (
        <div className="flex-grow flex flex-col items-center justify-center p-2 sm:p-4 bg-gray-900 text-white">
            <h1 className="text-4xl font-bold mb-4">♟️ شطرنج</h1>
            
            <div className="flex flex-col lg:flex-row items-start gap-8 w-full max-w-6xl mx-auto">
                {/* Chessboard and Game Over Panel */}
                <div className="relative lg:order-1 flex-grow flex justify-center">
                    <Chessboard
                        // FIX: The `react-chessboard` component uses the `position` prop to accept a FEN string, not `fen`.
                        position={fen}
                        onPieceDrop={onDrop}
                        boardWidth={boardWidth}
                        customBoardStyle={{
                            borderRadius: '8px',
                            boxShadow: '0 5px 15px rgba(0, 0, 0, 0.5)',
                        }}
                    />
                     {gameOver && (
                        <div className="absolute inset-0 bg-black/70 flex flex-col items-center justify-center rounded-lg">
                            <h2 className="text-3xl font-bold text-white mb-2">{gameOver.message}</h2>
                            <p className="text-xl text-yellow-400 mb-6">{gameOver.reason}</p>
                            <div className="flex gap-4">
                                <button onClick={playAgain} className="bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-3 px-8 rounded-full text-lg transition-transform transform hover:scale-105 shadow-lg">
                                    بازی مجدد
                                </button>
                                <Link to="/games" className="bg-gray-600 hover:bg-gray-700 text-white font-bold py-3 px-8 rounded-full text-lg transition-transform transform hover:scale-105 shadow-lg">
                                    خروج
                                </Link>
                            </div>
                        </div>
                    )}
                </div>

                {/* Game Info & Chat Panel */}
                <div className="w-full lg:w-80 flex-shrink-0 lg:order-2 space-y-4">
                    <div className="bg-gray-800 p-4 rounded-lg shadow-xl border border-gray-700">
                        <h3 className="text-xl font-bold mb-4 border-b border-gray-600 pb-2">اطلاعات بازی</h3>
                        <div className="space-y-3 mb-4">
                            <p className="text-lg">
                                <span className="font-bold text-gray-300">سفید: </span> 
                                {auth.user.username} (شما)
                            </p>
                            <p className="text-lg">
                                <span className="font-bold text-gray-800 bg-white rounded-sm px-1">سیاه: </span> 
                                {opponent.username}
                            </p>
                        </div>
                        <div className="text-lg font-semibold bg-gray-700 px-4 py-2 rounded-lg shadow-md min-h-[50px] flex items-center justify-center">
                            {turnStatus}
                        </div>
                    </div>
                    <Chat currentUser={auth.user} opponent={opponent} />
                </div>
            </div>
             {!isBotGame && !gameOver && (
                <div className="mt-6 bg-yellow-800/50 text-yellow-300 border border-yellow-700 rounded-md p-4 text-center max-w-md">
                    <p className="font-bold">حالت نمایشی</p>
                    <p className="text-sm">در حال حاضر فقط بازی با کامپیوتر فعال است. حالت دو نفره (انسان با انسان) در آینده اضافه خواهد شد.</p>
                </div>
            )}
        </div>
    );
};

export default ChessGame;