import React, { useState, useEffect, useContext } from 'react';
import { useNavigate, useParams, Link } from 'react-router-dom';
import { AuthContext } from '../App';
import type { User } from '../types';
import { getUsers } from '../services/storageService';
import { BOT_USER } from '../constants';

const gameTitles: { [key: string]: string } = {
    tictactoe: 'دوز (Tic-Tac-Toe)',
    ludo: 'منچ',
    chess: 'شطرنج',
    backgammon: 'تخته نرد'
};


const GameLobby: React.FC = () => {
  const [opponents, setOpponents] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const auth = useContext(AuthContext);
  const navigate = useNavigate();
  const { gameType } = useParams<{ gameType: string }>();
  
  const currentGameTitle = gameType ? gameTitles[gameType] : 'بازی';

  useEffect(() => {
    // Reset otherUser when entering a new lobby
    auth?.setOtherUser(null);
    
    const allUsers = getUsers();
    // Filter out the current user
    const availableOpponents = allUsers.filter(u => u.id !== auth?.user?.id);
    setOpponents(availableOpponents);
    setLoading(false);
  }, [auth, gameType]);


  const handleSelectOpponent = (opponent: User) => {
    if (auth && gameType) {
        auth.setOtherUser(opponent);
        // Pass opponent data directly via route state to prevent race conditions.
        // This ensures the game page has the opponent data on its initial render.
        navigate(`/game/${gameType}`, { state: { opponent } });
    }
  };
  
  if (!gameType || !gameTitles[gameType]) {
    return (
        <div className="flex-grow text-center p-8">
            <h2 className="text-2xl text-red-500">بازی نامعتبر است.</h2>
            <Link to="/games" className="text-indigo-400 hover:underline mt-4 inline-block">بازگشت به لیست بازی‌ها</Link>
        </div>
    );
  }

  if (loading) {
    return <div className="flex-grow text-center p-8">در حال بارگذاری لابی...</div>;
  }

  return (
    <div className="flex-grow container mx-auto p-4 md:p-8">
      <div className="bg-gray-800 p-6 rounded-lg shadow-2xl border border-gray-700 max-w-2xl mx-auto">
        <h1 className="text-4xl font-bold mb-2 text-center text-white">{`لابی بازی ${currentGameTitle}`}</h1>
        <p className="text-center text-gray-400 mb-8">
          یک بازیکن را برای شروع بازی انتخاب کنید.
        </p>

        {/* Bot Opponent */}
        <div className="mb-6">
           <button
                onClick={() => handleSelectOpponent(BOT_USER)}
                className="w-full text-left bg-teal-800/50 hover:bg-teal-600/50 border border-teal-600 p-4 rounded-lg flex items-center justify-between transition-all duration-200"
            >
                <div className="flex items-center">
                    <span className="text-xl font-semibold text-white">{BOT_USER.username}</span>
                    <span className="text-xs font-medium mr-3 bg-teal-500 text-white py-1 px-2 rounded-full">هوش مصنوعی</span>
                </div>
                <span className="text-sm font-bold py-2 px-4 rounded-full bg-teal-600 text-white shadow-md">
                    شروع بازی
                </span>
            </button>
        </div>

        <div className="relative my-4">
          <div className="absolute inset-0 flex items-center" aria-hidden="true">
            <div className="w-full border-t border-gray-600" />
          </div>
          <div className="relative flex justify-center">
            <span className="bg-gray-800 px-2 text-sm text-gray-400">یا یک بازیکن واقعی</span>
          </div>
        </div>

        {/* Human Opponents */}
        {opponents.length > 0 ? (
          <div className="space-y-4">
            {opponents.map(opponent => (
              <button
                key={opponent.id}
                onClick={() => handleSelectOpponent(opponent)}
                className="w-full text-left bg-gray-700/50 hover:bg-indigo-600/50 border border-gray-600 p-4 rounded-lg flex items-center justify-between transition-all duration-200"
              >
                <span className="text-xl font-semibold text-white">{opponent.username}</span>
                <span className="text-sm font-bold py-2 px-4 rounded-full bg-indigo-600 text-white shadow-md">
                  شروع بازی
                </span>
              </button>
            ))}
          </div>
        ) : (
          <p className="text-center text-gray-500 bg-gray-700/30 p-4 rounded-md">
            بازیکن دیگری برای بازی پیدا نشد. می‌توانید با کامپیوتر بازی کنید!
          </p>
        )}
      </div>
    </div>
  );
};

export default GameLobby;