import React, { useState, useEffect, useContext, useCallback } from 'react';
import { useNavigate, useLocation } from 'react-router-dom';
import Board from '../components/Board';
import { AuthContext } from '../App';
import { useScoreUpdater } from '../hooks/useScoreUpdater';
import { BOT_USER } from '../constants';
import type { User } from '../types';
import Chat from '../components/Chat';

// AI Helper: Checks for a winner and returns the symbol.
const checkWinnerForAI = (squares: Array<'X' | 'O' | null>): 'X' | 'O' | null => {
  const lines = [
    [0, 1, 2], [3, 4, 5], [6, 7, 8],
    [0, 3, 6], [1, 4, 7], [2, 5, 8],
    [0, 4, 8], [2, 4, 6],
  ];
  for (let i = 0; i < lines.length; i++) {
    const [a, b, c] = lines[i];
    if (squares[a] && squares[a] === squares[b] && squares[a] === squares[c]) {
      return squares[a];
    }
  }
  return null;
};

// The minimax algorithm for determining the best move for the AI.
const minimax = (board: Array<'X' | 'O' | null>, depth: number, isMaximizing: boolean): number => {
    const winner = checkWinnerForAI(board);

    if (winner === 'O') return 10 - depth; // AI ('O') wins - prefer faster wins.
    if (winner === 'X') return -10 + depth; // Player ('X') wins - prefer delaying losses.
    if (!board.includes(null)) return 0; // It's a draw.

    if (isMaximizing) { // AI's turn (Maximizer 'O')
        let bestScore = -Infinity;
        for (let i = 0; i < 9; i++) {
            if (board[i] === null) {
                board[i] = 'O';
                let score = minimax(board, depth + 1, false);
                board[i] = null; // Backtrack
                bestScore = Math.max(score, bestScore);
            }
        }
        return bestScore;
    } else { // Player's turn (Minimizer 'X')
        let bestScore = Infinity;
        for (let i = 0; i < 9; i++) {
            if (board[i] === null) {
                board[i] = 'X';
                let score = minimax(board, depth + 1, true);
                board[i] = null; // Backtrack
                bestScore = Math.min(score, bestScore);
            }
        }
        return bestScore;
    }
};

const findBestMove = (board: Array<'X' | 'O' | null>): number => {
    let bestScore = -Infinity;
    let move = -1;

    for (let i = 0; i < 9; i++) {
        if (board[i] === null) {
            board[i] = 'O'; // AI's move
            let score = minimax(board.slice(), 0, false); // find score for this move
            board[i] = null; // Backtrack
            if (score > bestScore) {
                bestScore = score;
                move = i;
            }
        }
    }
    return move;
};


const calculateWinner = (squares: Array<'X' | 'O' | null>): { winner: 'X' | 'O' | null, line: number[] | null } => {
  const lines = [
    [0, 1, 2],
    [3, 4, 5],
    [6, 7, 8],
    [0, 3, 6],
    [1, 4, 7],
    [2, 5, 8],
    [0, 4, 8],
    [2, 4, 6],
  ];
  for (let i = 0; i < lines.length; i++) {
    const [a, b, c] = lines[i];
    if (squares[a] && squares[a] === squares[b] && squares[a] === squares[c]) {
      return { winner: squares[a], line: lines[i] };
    }
  }
  return { winner: null, line: null };
};


const TicTacToeGame: React.FC = () => {
    const [history, setHistory] = useState([Array(9).fill(null)]);
    const [currentMove, setCurrentMove] = useState(0);
    const xIsNext = currentMove % 2 === 0;
    const currentSquares = history[currentMove];
    const { winner, line: winnerLine } = calculateWinner(currentSquares);
    const isDraw = !winner && currentSquares.every(Boolean);

    const auth = useContext(AuthContext);
    const navigate = useNavigate();
    const location = useLocation();
    const { updateScores } = useScoreUpdater();

    // Get opponent from route state first to prevent race condition, fallback to context.
    const opponent = (location.state?.opponent as User | undefined) || auth?.otherUser;

    useEffect(() => {
        // If opponent data exists, ensure it's synced with the global context.
        if (opponent && (!auth?.otherUser || auth.otherUser.id !== opponent.id)) {
            auth?.setOtherUser(opponent);
        }
    }, [opponent, auth]);

    const isBotTurn = !xIsNext && opponent?.id === BOT_USER.id;

    useEffect(() => {
        if (auth && !auth.isLoggedIn) {
            navigate('/login');
        }
    }, [auth, navigate]);

    useEffect(() => {
        if ((winner || isDraw) && auth?.user && opponent) {
            if (winner === 'X') { // 'X' is always auth.user (player1)
                updateScores(auth.user, opponent, 'player1_wins');
            } else if (winner === 'O') { // 'O' is always opponent (player2)
                updateScores(auth.user, opponent, 'player2_wins');
            } else if (isDraw) {
                updateScores(auth.user, opponent, 'draw');
            }
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [winner, isDraw, auth?.user, opponent]);

     const handlePlay = useCallback((nextSquares: Array<'X' | 'O' | null>) => {
        const nextHistory = [...history.slice(0, currentMove + 1), nextSquares];
        setHistory(nextHistory);
        setCurrentMove(nextHistory.length - 1);
    }, [history, currentMove]);

    const handleClick = useCallback((i: number) => {
        if (winner || currentSquares[i] || isBotTurn) {
            return;
        }
        const nextSquares = currentSquares.slice();
        nextSquares[i] = xIsNext ? 'X' : 'O';
        handlePlay(nextSquares);
    }, [winner, currentSquares, isBotTurn, xIsNext, handlePlay]);

    // Bot move logic
    useEffect(() => {
        if (isBotTurn && !winner && !isDraw) {
            const timer = setTimeout(() => {
                // Use a copy for the AI algorithm as it mutates the board.
                const boardCopy = currentSquares.slice();
                const bestMove = findBestMove(boardCopy);
                if (bestMove !== -1) {
                    // Create a new board state for the update.
                    const nextSquares = currentSquares.slice();
                    // The bot is always 'O'.
                    nextSquares[bestMove] = 'O';
                    // Directly call handlePlay to avoid the guard in handleClick.
                    handlePlay(nextSquares);
                }
            }, 800); // Bot "thinking" delay
            return () => clearTimeout(timer);
        }
    }, [isBotTurn, winner, isDraw, currentSquares, handlePlay]);


    const resetGame = () => {
        setHistory([Array(9).fill(null)]);
        setCurrentMove(0);
    };

    let status;
    if (winner) {
        const winnerUsername = winner === 'X' ? auth?.user?.username : opponent?.username;
        status = `برنده: ${winnerUsername} (${winner})`;
    } else if (isDraw) {
        status = 'بازی مساوی شد!';
    } else {
        const nextPlayerUsername = xIsNext ? auth?.user?.username : opponent?.username;
        status = `نوبت بازیکن: ${nextPlayerUsername} (${xIsNext ? 'X' : 'O'})`;
        if (isBotTurn) {
            status = 'کامپیوتر در حال فکر کردن است...';
        }
    }

    if (!auth?.isLoggedIn || !auth.user || !opponent) {
        return <div className="flex-grow flex items-center justify-center text-center p-8">در حال آماده‌سازی بازی...</div>
    }

    return (
        <div className="flex-grow flex flex-col items-center justify-center p-4 bg-gray-900 text-white">
            <h1 className="text-4xl font-bold mb-4">بازی دوز (Tic-Tac-Toe)</h1>
            <div className="flex flex-col md:flex-row items-center md:items-start gap-8">
                <div className="flex flex-col items-center">
                    <div className="text-2xl font-semibold mb-4 bg-gray-800 px-6 py-3 rounded-lg shadow-md min-h-[60px] flex items-center justify-center">{status}</div>
                    <Board squares={currentSquares} onClick={handleClick} winnerLine={winnerLine} />
                     {(winner || isDraw) && (
                        <button onClick={resetGame} className="mt-6 bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-3 px-8 rounded-full text-lg transition-transform transform hover:scale-105 shadow-lg">
                            بازی مجدد
                        </button>
                    )}
                </div>
                <div className="w-full md:w-80 space-y-4">
                    <div className="bg-gray-800 p-6 rounded-lg shadow-xl border border-gray-700">
                        <h3 className="text-xl font-bold mb-4 border-b border-gray-600 pb-2">بازیکنان</h3>
                        <div className="space-y-3">
                            <p className="text-lg">
                                <span className="font-bold text-sky-400">X: </span> 
                                {auth.user.username} (شما)
                            </p>
                            <p className="text-lg">
                                <span className="font-bold text-amber-400">O: </span> 
                                {opponent.username}
                            </p>
                        </div>
                    </div>
                     <Chat currentUser={auth.user} opponent={opponent} />
                </div>
            </div>
        </div>
    );
};

export default TicTacToeGame;