const express = require('express');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const path = require('path');

const app = express();
const server = http.createServer(app);

// تنظیمات CORS
const io = socketIo(server, {
  cors: {
    origin: "*",
    methods: ["GET", "POST"]
  }
});

app.use(cors());
app.use(express.static('public'));

// ذخیره اطلاعات کاربران آنلاین
const users = new Map();
const rooms = new Map();

// رویدادهای Socket.IO
io.on('connection', (socket) => {
  console.log(`کاربر جدید متصل شد: ${socket.id}`);

  // پیوستن به اتاق
  socket.on('join-room', (roomId, userName) => {
    try {
      // اضافه کردن کاربر به اتاق
      socket.join(roomId);
      
      // ذخیره اطلاعات کاربر
      users.set(socket.id, {
        id: socket.id,
        name: userName || `کاربر_${socket.id.slice(0, 6)}`,
        roomId: roomId,
        isMuted: false
      });

      // ذخیره اطلاعات اتاق
      if (!rooms.has(roomId)) {
        rooms.set(roomId, new Set());
      }
      rooms.get(roomId).add(socket.id);

      const user = users.get(socket.id);
      console.log(`${user.name} وارد اتاق ${roomId} شد`);

      // اطلاع‌رسانی به سایر کاربران اتاق
      socket.to(roomId).emit('user-joined', {
        userId: socket.id,
        userName: user.name,
        participantCount: rooms.get(roomId).size
      });

      // ارسال لیست کاربران فعلی به کاربر جدید
      const roomUsers = Array.from(rooms.get(roomId))
        .map(id => users.get(id))
        .filter(user => user);

      socket.emit('room-users', roomUsers);
      
      // ارسال تعداد شرکت‌کنندگان
      io.to(roomId).emit('participant-count', rooms.get(roomId).size);

    } catch (error) {
      console.error('خطا در پیوستن به اتاق:', error);
      socket.emit('error', 'خطا در پیوستن به اتاق');
    }
  });

  // مدیریت پیشنهادات WebRTC
  socket.on('offer', (data) => {
    socket.to(data.target).emit('offer', {
      offer: data.offer,
      sender: socket.id
    });
  });

  socket.on('answer', (data) => {
    socket.to(data.target).emit('answer', {
      answer: data.answer,
      sender: socket.id
    });
  });

  socket.on('ice-candidate', (data) => {
    socket.to(data.target).emit('ice-candidate', {
      candidate: data.candidate,
      sender: socket.id
    });
  });

  // مدیریت bی‌صدا کردن
  socket.on('toggle-mute', () => {
    const user = users.get(socket.id);
    if (user) {
      user.isMuted = !user.isMuted;
      io.to(user.roomId).emit('user-muted', {
        userId: socket.id,
        isMuted: user.isMuted
      });
      console.log(`${user.name} ${user.isMuted ? 'بی‌صدا شد' : 'صدادار شد'}`);
    }
  });

  // قطع اتصال
  socket.on('disconnect', () => {
    const user = users.get(socket.id);
    
    if (user) {
      console.log(`${user.name} اتصال را قطع کرد`);
      
      // حذف از اتاق
      if (rooms.has(user.roomId)) {
        rooms.get(user.roomId).delete(socket.id);
        
        // اطلاع‌رسانی به سایر کاربران
        socket.to(user.roomId).emit('user-left', {
          userId: socket.id,
          userName: user.name,
          participantCount: rooms.get(user.roomId).size
        });

        // ارسال تعداد جدید شرکت‌کنندگان
        io.to(user.roomId).emit('participant-count', rooms.get(user.roomId).size);

        // حذف اتاق در صورت خالی بودن
        if (rooms.get(user.roomId).size === 0) {
          rooms.delete(user.roomId);
          console.log(`اتاق ${user.roomId} حذف شد`);
        }
      }
      
      // حذف کاربر
      users.delete(socket.id);
    }
  });

  // ارسال پیام چت (اختیاری)
  socket.on('chat-message', (message) => {
    const user = users.get(socket.id);
    if (user) {
      io.to(user.roomId).emit('chat-message', {
        userName: user.name,
        message: message,
        timestamp: new Date().toISOString()
      });
    }
  });
});

// روت اصلی
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// API برای دریافت آمار
app.get('/api/stats', (req, res) => {
  res.json({
    totalUsers: users.size,
    totalRooms: rooms.size,
    rooms: Array.from(rooms.entries()).map(([roomId, userIds]) => ({
      roomId,
      userCount: userIds.size
    }))
  });
});

const PORT = process.env.PORT || 9969;
server.listen(PORT, () => {
  console.log(`🎤 سرور ویس چت در حال اجرا: http://localhost:${PORT}`);
  console.log(`📊 آمار: http://localhost:${PORT}/api/stats`);
});